##Copyright 2011 William Miller	-	kronos37@gmail.com
##
##This file is part of Sudoku.
##
##Sudoku is free software: you can redistribute it and/or modify
##it under the terms of the GNU General Public License as published by
##the Free Software Foundation, either version 3 of the License, or
##(at your option) any later version.
##
##Sudoku is distributed in the hope that it will be useful,
##but WITHOUT ANY WARRANTY; without even the implied warranty of
##MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##GNU General Public License for more details.
##
##You should have received a copy of the GNU General Public License
##along with Sudoku.  If not, see <http://www.gnu.org/licenses/>.

#class definition for sudoku
import pickle

class Sudoku:
	def __init__(self, newSize):
		"""
			Initiates the Sudoku puzzle with the size as given by newSize
			Given the size of the puzzle, marks the size of the boxes inside the Sudoku
			param: newSize the size of the puzzle

			self.SIZE stores the width of the Sudoku
			self.BOX_WIDTH stores the width of the internal boxes of the Sudoku
			self.puzzle is the version of the puzzle to be operated on and changed
			access to particular squares is given by self.puzzle[row][column]=value
			self.puzzleRowColNum is the traditional representation of the Sudoku
			self.puzzleRowNumCol is a representation with Row on the y axis, Number of the x axis and Columns in the field
			access is given by self.puzzleRowNumCol[row][value]=column
			self.puzzleNumColRow is a representation with Numbers on the y axis, Columns on the x axis and Rows in the field
			access is given by self.puzzleNumColRow[value][column]=row
			self.possible stores the possible values for each square or cell in the puzzle
			access is given by self.possible[row][col]=[possible values]
			self.repMode marks which version of the puzzle is being displayed and stored in self.puzzle
			repMode 1 is RowColNum, repMode 2 is RowNumCol and repMode 3 is NumColRow
			self.manualPossible is a marker to show whether or not the possibilities are generated by the machine or added by the user,
			True is user possibilities
			self.operationList stores the operations done on the puzzle so an undo function can be implemented
			each operation added to operationList will be a list of [row, col, lastValue, newValue, repMode]
		"""
		if newSize == 4 or newSize == 9 or newSize == 16:
			self.SIZE = newSize
			#Size marker for puzzle
			self.puzzle = [[0 for row in range(self.SIZE)]for col in range(self.SIZE)]
			#the puzzle to be operated on in which ever representation desired
			self.possible = [[[]for row in range(self.SIZE)]for col in range(self.SIZE)]
			#the possible values for each square in puzzle
			self.puzzleRowNumCol = [[0 for row in range(self.SIZE)]for col in range(self.SIZE)]
			#Row, Number, Column representation of puzzle
			self.puzzleNumColRow = [[0 for row in range(self.SIZE)]for col in range(self.SIZE)]
			#Number, Column, Row representation of puzzle
			self.puzzleRowColNum = [[0 for row in range(self.SIZE)]for col in range(self.SIZE)]
			#Traditional Row, Column, Number representation of puzzle
			self.repMode = 1
			#marks which version of the puzzle is being displayed in self.puzzle
			self.manualPossible = False
			#marks whether or not automatic possibility generating is on
			self.operationList = []
			if self.SIZE == 4:      #box width marker for various different sizes
				self.BOX_WIDTH = 2
			elif self.SIZE == 9:
				self.BOX_WIDTH = 3
			elif self.SIZE == 16:
				self.BOX_WIDTH = 4
		else:
			raise Exception("Invalid size")

	def addValue(self, row, col, value):
		"""
			Adds a value to the printable puzzle
			param: row the row of the square to be added to
			param: col the column of the square to be added to
			param: value the value to be added
		"""
		#makes sure the values passed it are within the range, then adds the value to the puzzle
		if row >= 0 and row < self.SIZE and col >= 0 and col < self.SIZE and value > 0 and value <= self.SIZE:
			if self.repMode == 1:
				lastValue = self.puzzle[row][col]
				self.puzzle[row][col] = value
				self.operationList.append([row, col, lastValue, value, self.repMode])
			elif self.repMode == 2:
				for x in range(self.SIZE):
					if value == self.puzzle[row][x]:
						raise Exception("Can not assign two values to one cell")
				lastValue = self.puzzle[row][col]
				self.puzzle[row][col] = value
				self.operationList.append([row, col, lastValue, value, self.repMode])
			else:
				#for x in range(self.SIZE):
					#if value == self.puzzle[x][col]:
					#	raise Exception("Can not assign two values to one cell")
				lastValue = self.puzzle[row][col]
				self.puzzle[row][col] = value
				self.operationList.append([row, col, lastValue, value, self.repMode])
		else:
			raise Exception("Invalid Values")

	def deleteValue(self, row, col):
		"""
			Resets a squares value back to 0
			param: row the row of the square to be reset
			param: col the column of the square to be reset
		"""
		#makes sure the position of the square is valid, then resests the value of the square to 0
		if row >= 0 and row < self.SIZE and col >= 0 and col < self.SIZE:
			lastValue = self.puzzle[row][col]
			self.puzzle[row][col] = 0
			self.operationList.append([row, col, lastValue, 0, self.repMode])
		else:
			raise Exception("Invalid Values")

	def validateValue(self, row, col, value):
		"""
			Checks if a value can be added to a cell by looking for duplicate values
			Returns all squares with duplicate values in area of origial square
			Returns an empty list if there are no duplicate values
		"""
		squareList = []
		for n in range(self.SIZE):
			if value == self.puzzle[n][col]:	#looks for duplicate value in row
				squareList.append([n,col])
			if value == self.puzzle[row][n]:	#looks for duplicate value in column
				squareList.append([row,n])
		for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
			if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
				x = i * self.BOX_WIDTH
			if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
				y = i * self.BOX_WIDTH
		xCount = x
		xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
		yMax = y + self.BOX_WIDTH - 1
		while (xCount <= xMax): #deals with the box the square is in
			yCount = y
			while (yCount <= yMax):
				if value == self.puzzle[x][y]:	#looks for duplicate values in box
					squareList.append([x,y])
				yCount = yCount + 1
			xCount = xCount + 1
		return squareList

	def modifyPossible(self, row, col, value):
		"""
			Modifies the possible values if manualPossible is set to true
			If the value is present in the possibilities, it removes that possibility
			If the value is not present in the possiblilties, it adds that possibility
		"""
		if self.manualPossible:
			if row >= 0 and row < self.SIZE and col >= 0 and col < self.SIZE and value > 0 and value <= self.SIZE:
				if self.possible[row][col].count(value) == 1:
					self.possible[row][col].remove(value)
				else:
					self.possible[row][col].append(value)
					self.possible[row][col].sort()
			else:
				raise Exception("Invalid values")

	def updatePossibleSquare(self, row, col):
		"""
			Updates the possible values of all squares associated with a given square
		"""
		if not self.manualPossible:
			for n in range(self.SIZE):	#updates the row and column
				self.findPossibleSquare(n, col)
				self.findPossibleSquare(row, n)
			#defines the edges of the box
			row2 = row + (self.BOX_WIDTH - 1) 		#Defines size of box to check
			col2 = col + (self.BOX_WIDTH - 1)
			x = row
			while (x <= row2):	#Cycles through the box
				y = col
				while (y <= col2):      #Finds the possible values for the box
					self.findPossibleSquare(x, y)
					y = y + 1
				x = x + 1

	def printPuzzle(self):
		"""
			Prints the puzzle for the console version
		"""
		count2 = 0
		if self.BOX_WIDTH == 2:
			print " -----------"
		elif self.BOX_WIDTH == 3:
			print " -----------------------"
		for row in self.puzzle:
			rowString = "| "
			count = 0
			for x in row:
				if x == 0:
					rowString = rowString + ". "
				else:
					rowString = rowString + str(x) + " "
				count = count + 1
				if count % self.BOX_WIDTH == 0:
					rowString = rowString + "| "
			print rowString
			count2 = count2 + 1
			if count2 % self.BOX_WIDTH == 0:
				if self.BOX_WIDTH == 2:
					print " -----------"
				elif self.BOX_WIDTH == 3:
					print " -----------------------"

	def printPossiblePuzzle(self):	#needs proper formating
		"""
			Prints the puzzle, including possible values, for the console version
		"""
		count2 = 0
##		if self.BOX_WIDTH == 2:
##			print " ----------------------------------"
##		elif self.BOX_WIDTH == 3:
##			print " ----------------------------------------------------------------"
		for row in range(self.SIZE):
			rowString = "| "
			for col in range(self.SIZE):
				possib = self.possible[row][col]
				if possib == []:
					rowString = rowString + str(self.puzzle[row][col]) + "\t"
				else:
					rowString = rowString + str(possib) + "\t"
				if (col + 1) % self.BOX_WIDTH == 0:
					rowString = rowString + "| "
			print rowString
##			if (row + 1) % self.BOX_WIDTH == 0:
##				if self.BOX_WIDTH == 2:
##					print " ----------------------------"
##				elif self.BOX_WIDTH == 3:
##					print " --------------------------------------------------------------"

	def checkRow(self, row):
		"""
			Checks the given column for validity in terms of a Sudoku puzzle
			Looks for duplicates and missing values
			param: col the index of the column to be checked
			post: Will return false if there are any values which have been duplicated
		"""
		check = range(self.SIZE + 1)	#use N+1 where N is size of puzzle
		valid = True
		for col in range(self.SIZE):    #finds the value of the square in each column of the row and removes the value from check
			value = self.puzzleRowColNum[row][col]
			check[value] = check[value] - value
		for x in range(self.SIZE + 1):  #if there are any negative values in check, means something has been removed twice, will return false
			if check[x] < 0:
				valid = False
		return valid

	def checkCol(self, col):
		"""
			Checks the given column for validity in terms of a Sudoku puzzle
			Looks for duplicates and missing values
			param: col the index of the column to be checked
			post: Will return false if there are any values which have been duplicated
		"""
		check = range(self.SIZE + 1)	#use N+1 where N is size of puzzle
		valid = True
		for row in range(self.SIZE):    #finds the value of the square in each row of the column and removes the value from check
			value = self.puzzleRowColNum[row][col]
			check[value] = check[value] - value
		for x in range(self.SIZE + 1):  #if there are any negative values in check, will return false
			if check[x] < 0:
				valid = False
		return valid

	def checkBox(self, row, col):
		""" 
			Checks the given box for validity in terms of a Sudoku puzzle
			Looks for duplicates and missing values
			param: row the row index of the top left cell of box
			param: col the column index of the top left cell of box
			pre: requires the square passed to the method is the top left square of the box
			post: returns false if any value in the box has been duplicated
		"""
		check = range(self.SIZE + 1)
		valid = True
		row2 = row + (self.BOX_WIDTH - 1) 		#Defines size of box to check
		col2 = col + (self.BOX_WIDTH - 1)
		x = row
		while (x <= row2):	#Cycles through the box
			y = col
			while (y <= col2):      #finds the value in each of the squares of the box and removes them from check
				value = self.puzzleRowColNum[x][y]
				check[value] = check[value] - value
				y = y + 1
			x = x + 1
		for n in range(self.SIZE + 1):  #if there are any negative values in check, returns false
			if check[n] < 0:
				valid = False
		return valid
	

	def checkPuzzle(self):
		"""
			Checks the entire puzzle for invalid rows, columns and boxes
			Checks the puzzle to make sure it is valid
			Returns invalidList, which is a list of all invalid positions
			if invalidList is empty, the puzzle is valid
			an entry of [1,x] indicates a row is invalid and x is the index of that row
			an entry of [2,x] indicates a column is invalid and x is the index of that column
			an entry of [3,x,y] indicates a box is invalid and x is the row index of the top left square of that box
			while y is the column index of the top left square of the box
		"""
		self.updatePuzzle()
		validRow = True
		validCol = True
		validBox = True      #Markers for each different validation method
		invalidList = []		#Stores the invalid positions
		for n in range(self.SIZE):	#Checks each row and column of Sudoku
			validRow = self.checkRow(n)
			validCol = self.checkCol(n)
			if not validRow:
				invalidList.append([1,n])
			if not validCol:
				invalidList.append([2,n])
		x = 0
		while (x < (self.SIZE - 1)):		#Checks the different boxes of Sudoku
			y = 0
			while (y < (self.SIZE - 1)):	#while x and y are less than index of last position of list
				validBox = self.checkBox(x, y)
				if not validBox:
					invalidList.append([3,x,y])
				y = y + (self.BOX_WIDTH)
			x = x + (self.BOX_WIDTH)
		return invalidList

	def isComplete(self):
		"""
			Checks to see if the puzzle is complete
			Returns true if every square has a value
			Returns false otherwise
		"""
		complete = True 	#marker for whether puzzle is complete or not
		for row in range(self.SIZE):	#checks every square
			for col in range(self.SIZE):	
				if not self.puzzle[row][col] > 0:	#if there is no value, breaks the loop and marks the puzzle as incomplete
					complete = False
					break
		return complete

	def findPossibleSquare(self, row, col):
		"""
			Checks a given square for possibilities and adds the possibilities to the possible list
		"""
		self.possible[row][col] = []	#resets the possible list for the given square
		if self.puzzle[row][col] == 0:
			rowCheck = range(self.SIZE + 1)
			colCheck = range(self.SIZE + 1)
			boxCheck = range(self.SIZE + 1) 
			for n in range(self.SIZE):	#deals with the column and row associated with square
				rowValue = self.puzzle[row][n]
				colValue = self.puzzle[n][col]
				rowCheck[rowValue] = rowCheck[rowValue] - rowValue
				colCheck[colValue] = colCheck[colValue] - colValue
			if self.repMode == 1:	#only deals with boxes if the puzzle is in traditional representation mode
				for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
					if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
						x = i * self.BOX_WIDTH
					if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
						y = i * self.BOX_WIDTH
				xCount = x
				xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
				yMax = y + self.BOX_WIDTH - 1
				while (xCount <= xMax): #deals with the box the square is in
					yCount = y
					while (yCount <= yMax):
						boxValue = self.puzzle[xCount][yCount]
						boxCheck[boxValue] = boxCheck[boxValue] - boxValue
						yCount = yCount + 1
					xCount = xCount + 1
			for j in range(self.SIZE + 1):  #Adds the value to possible options if it is a possibility in the row, column and box
				if (rowCheck[j] == colCheck[j]) and (rowCheck[j] == boxCheck[j]) and (rowCheck[j] > 0):
					self.possible[row][col].append(j)

	def findPossiblePuzzle(self):
		"""
			Checks the entire puzzle for possibilities and completes the possible list
		"""
		if not self.manualPossible:     #only finds possible values if it is set to auto generate possible values
			for row in range(self.SIZE):    #calls findPossibleSquare for every square in the puzzle
				for col in range(self.SIZE):
					self.findPossibleSquare(row, col)

	def getPossibilities(self, row, col):
		"""
			Returns the possibilities of a given square as a string
		"""
		string = str(self.possible[row][col])
		string = string[1:string.__len__() - 1]
		return string
	
	def generateRowNumCol(self):
		"""
			Generates a representation of the puzzle with rows on the y axis, numbers on the x axis and columns in the field
			Uses puzzleRowColNum to generate, so puzzle must be updated before running method
		"""
		self.puzzleRowNumCol = [[0 for row in range(self.SIZE)] for col in range(self.SIZE)]	#resets the RowNumCol representation
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				value = self.puzzleRowColNum[row][col]
				if value > 0:
					self.puzzleRowNumCol[row][value-1] = col + 1

	def generateNumColRow(self):
		"""
			Generates a representaion of the puzzle with numbers on the y axis, columns on the x axis and rows in the field
			Uses puzzleRowColNum to generate, so puzzle must be updated before running method
		"""
		self.puzzleNumColRow = [[0 for row in range(self.SIZE)] for col in range(self.SIZE)]	#resets the NumColRow representation
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				value = self.puzzleRowColNum[row][col]
				if value > 0:
					self.puzzleNumColRow[value-1][col] = row + 1

	def savePuzzle(self, fileName):
		"""
			Saves the sudoku to the filePath file using pickle module
		"""
		try:
			temp = self.repMode
			self.changeRepMode(1)
			save = open(fileName, 'w')
			pickle.dump(self, save)
			save.close()
			self.changeRepMode(temp)
		except Exception as e:
			print e

	def findSingleCandidate(self):
		"""
			Searches the puzzle for any squares with only one possibility and returns these squares
			returns a list of squares, with each entry being [row,column]
		"""
		squareList = []
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				if self.possible[row][col].__len__() == 1:
					squareList.append([row,col])
		return squareList

	def findSinglePosition(self):
		"""
			Searches the puzzle for any value which only has
			one available position in a given box, row or col
			Returns a list of squares, with each entry being [row,column,value]
		"""
		squareList = []
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				for value in self.possible[row][col]:
					validRow = True
					validCol = True
					validBox = True
					for n in range(self.SIZE):
						if self.possible[n][col].count(value) > 0 and n != row:	#deals with the row the square is in
							validRow = False
						if self.possible[row][n].count(value) > 0 and n != col:	#deals with the column the square is in
							validCol = False
					for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
						if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
							x = i * self.BOX_WIDTH
						if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
							y = i * self.BOX_WIDTH
					xCount = x
					xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
					yMax = y + self.BOX_WIDTH - 1
					while (xCount <= xMax): #deals with the box the square is in
						yCount = y
						while (yCount <= yMax):
							if self.possible[xCount][yCount].count(value) > 0 and not (xCount == row and yCount == col):
								validBox = False
							yCount = yCount + 1
						xCount = xCount + 1
					if validRow or validCol or validBox:
						squareList.append([row,col,value])
		return squareList

	def findCandidateLine(self):
		"""
			Searches the puzzle for any candidate lines, where values are only possible in a line
			in a given box, thus allowing that value to be removed as a possibility in that line in other
			Returns a list of squares, with each entry being [type of candidate line, row, column, index of candidate line, value]
		"""
		squareList = []
		x = 0
		while (x < (self.SIZE - 1)):		#Checks the different boxes of Sudoku
			y = 0
			while (y < (self.SIZE - 1)):	#while x and y are less than index of last position of list
				for value in range(self.SIZE + 1):	#checks each possible value
					possibleRow = []
					possibleCol = []
					for n in range(x, x + self.BOX_WIDTH):
						for i in range(y, y + self.BOX_WIDTH):
							if self.possible[n][i].count(value) > 0:
								possibleRow.append(n)
							if self.possible[i][n].count(value) > 0:
								possibleCol.append(n)
					if possibleRow.__len__() > 0:
						if possibleRow.count(possibleRow[0]) == possibleRow.__len__():
							squareList.append([1, x, y, possibleRow[0], value])
					if possibleCol.__len__() > 0:
						if possibleCol.count(possibleCol[0]) == possibleCol.__len__():
							squareList.append([2, x, y, possibleCol[0], value])
				y = y + (self.BOX_WIDTH)
			x = x + (self.BOX_WIDTH)
		return squareList

	def findNakedPair(self):
		"""
			Searches the puzzle for a naked pair - two squares in an area with the only two possibilities that are both the same
			Returns a list of squares, with each entry being
			[row of square 1, column of square 1, row of square 2, column of square 2, the possible values]
		"""
		squareList = []
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				if self.possible[row][col].__len__() == 2:
					for n in range(self.SIZE):
						#checks the row for a pair
						if self.possible[row][col] == self.possible[row][n] and n != col:
							squareList.append([row, col, row, n, self.possible[row][col]])
						#checks the column for a pair
						if self.possible[row][col] == self.possible[n][col] and n != row:
							squareList.append([row, col, n, col, self.possible[row][col]])
					#checks the box for a pair
					if self.repMode == 1:
						for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
							if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
								x = i * self.BOX_WIDTH
							if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
								y = i * self.BOX_WIDTH
						xCount = x
						xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
						yMax = y + self.BOX_WIDTH - 1
						while (xCount <= xMax): #deals with the box the square is in
							yCount = y
							while (yCount <= yMax):
								if self.possible[row][col] == self.possible[xCount][yCount] and not (xCount == row and yCount == col):
									squareList.append([row, col, xCount, yCount, self.possible[row][col]])
								yCount = yCount + 1
							xCount = xCount + 1
		for ni in squareList:
			for ecky in squareList:
				if not ni == ecky:
					if ni[1] == ecky[3] and ni[0] == ecky[2] and ni[4] == ecky[4]:
						squareList.remove(ecky)
		return squareList

	def findNakedTriple(self):
		"""
			Searches the puzzle for a naked triple - three squares in an area with only three possible values between them
			Only works for a size greater than 4
			Returns a list with each entry being
			[row1, column1, row2, column2, row3, column3, the possible values]
		"""
		squareList = []
		if self.SIZE == 4:
			return squareList
		for row in range(self.SIZE):
			for col in range(self.SIZE):
				if self.possible[row][col].__len__() == 3:	#case 1: when the current square contains all elements of the triple
					possibleRow = []
					possibleCol = []
					possibleBox = []
					for n in range(self.SIZE):
						#checks the row for a pair
						if self.possible[row][col] == self.possible[row][n] and n != col:
							possibleRow.append([row, col, row, n, self.possible[row][n]])
						if self.possible[row][n].__len__() == 2:
							if set(self.possible[row][n]).issubset(self.possible[row][col]):
								possibleRow.append([row, col, row, n, self.possible[row][n]])
						#checks the column for a pair
						if self.possible[row][col] == self.possible[n][col] and n != row:
							possibleCol.append([row, col, n, col, self.possible[row][col]])
						if self.possible[n][col].__len__() == 2:
							if set(self.possible[n][col]).issubset(self.possible[row][col]):
								possibleCol.append([row, col, n, col, self.possible[n][col]])
					#checks the box for a pair
					if self.repMode == 1:
						for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
							if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
								x = i * self.BOX_WIDTH
							if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
								y = i * self.BOX_WIDTH
						xCount = x
						xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
						yMax = y + self.BOX_WIDTH - 1
						while (xCount <= xMax): #deals with the box the square is in
							yCount = y
							while (yCount <= yMax):
								if self.possible[row][col] == self.possible[xCount][yCount] and not (xCount == row and yCount == col):
									possibleBox.append([row, col, xCount, yCount, self.possible[row][col]])
								if self.possible[xCount][yCount].__len__() == 2:
									if set(self.possible[xCount][yCount]).issubset(self.possible[row][col]):
										   possibleBox.append([row, col, xCount, yCount, self.possible[row][col]])
								yCount = yCount + 1
							xCount = xCount + 1
					#sorts the possibles and adds the actual triples to the return list
					if possibleRow.__len__() == 2:
						entry = possibleRow[0][0:4] + possibleRow[1][2:4]
						values = set(possibleRow[0][4] + possibleRow[1][4])
						entry.append(list(values))
						squareList.append(entry)
					if possibleCol.__len__() == 2:
						entry = possibleCol[0][0:4] + possibleCol[1][2:4]
						values = set(possibleCol[0][4] + possibleCol[1][4])
						entry.append(list(values))
						squareList.append(entry)
					if possibleBox.__len__() == 2:
						entry = possibleBox[0][0:4] + possibleBox[1][2:4]
						values = set(possibleBox[0][4] + possibleBox[1][4])
						entry.append(list(values))
						squareList.append(entry)

				if self.possible[row][col].__len__() == 2:	#case 2: When the current square contains only 2 of the elements of the triple
					possibleRow = []
					possibleCol = []
					possibleBox = []
					for n in range(self.SIZE):
						#checks the row for a pair
						if self.possible[row][n].__len__() == 3:
							if set(self.possible[row][col]).issubset(self.possible[row][n]):
								possibleRow.append([row, col, row, n, self.possible[row][n]])
						if self.possible[row][n].__len__() == 2:
							if self.possible[row][col] != self.possible[row][n]:
								for i in self.possible[row][col]:
									for j in self.possible[row][n]:
										if i == j:
											possibleRow.append([row, col, row, n, self.possible[row][n]])
											break
						#checks the column for a pair
						if self.possible[n][col].__len__() == 3:
							if set(self.possible[row][col]).issubset(self.possible[n][col]):
								possibleCol.append([row, col, n, col, self.possible[n][col]])
						if self.possible[n][col].__len__() == 2:
							if self.possible[row][col] != self.possible[n][col]:
								for i in self.possible[row][col]:
									for j in self.possible[n][col]:
										if i == j:
											possibleCol.append([row, col, n, col, self.possible[n][col]])
											break
					#checks the box for a pair
					if self.repMode == 1:
						for i in range(self.BOX_WIDTH + 1):     #finds which box the square is in
							if row >= (i*self.BOX_WIDTH) and row < ((i + 1)*self.BOX_WIDTH):
								x = i * self.BOX_WIDTH
							if col >= (i*self.BOX_WIDTH) and col < ((i + 1)*self.BOX_WIDTH):
								y = i * self.BOX_WIDTH
						xCount = x
						xMax = x + self.BOX_WIDTH - 1   #limits the squares to be checked to the box
						yMax = y + self.BOX_WIDTH - 1
						while (xCount <= xMax): #deals with the box the square is in
							yCount = y
							while (yCount <= yMax):
								if self.possible[xCount][yCount].__len__() == 2:
									if self.possible[row][col] != self.possible[xCount][xCount]:
										for i in self.possible[row][col]:
											for j in self.possible[xCount][yCount]:
												if i == j:
													possibleBox.append([row, col, xCount, yCount, self.possible[xCount][yCount]])
													break
								if self.possible[xCount][yCount].__len__() == 3:
									if set(self.possible[row][col]).issubset(self.possible[xCount][yCount]):
										   possibleBox.append([row, col, xCount, yCount, self.possible[xCount][yCount]])
								yCount = yCount + 1
							xCount = xCount + 1
					#Sorts the possibles and add the triples to the return list
					if possibleRow.__len__() == 2:
						entry = possibleRow[0][0:4] + possibleRow[1][2:4]
						values = set(possibleRow[0][4] + possibleRow[1][4])
						if values.__len__() == 3:
							entry.append(list(values))
							squareList.append(entry)
					if possibleCol.__len__() == 2:
						entry = possibleCol[0][0:4] + possibleCol[1][2:4]
						values = set(possibleCol[0][4] + possibleCol[1][4])
						if values.__len__() == 3:
							entry.append(list(values))
							squareList.append(entry)
					if possibleBox.__len__() == 2:
						entry = possibleBox[0][0:4] + possibleBox[1][2:4]
						values = set(possibleBox[0][4] + possibleBox[1][4])
						if values.__len__() == 3:
							entry.append(list(values))
							squareList.append(entry)
					if possibleRow.__len__() > 2:
						valueSet = []
						for n in possibleRow:
							if n[4].__len__() == 3:
								valueSet = n[4]
								break
						if valueSet.__len__() != 3:
							values = []
							for m in range(possibleRow.__len__()):
								values = values + possibleRow[m][4]
							valueSet = set(values)
							if valueSet.__len__() != 3:
								newSet = []
								for i in valueSet:
									newSet.append(i)
									if values.count(i) < 2:
										newSet.remove(i)
								valueSet = newSet
						if valueSet.__len__() == 3:
							for e in possibleRow:
								if not set(e[4]).issubset(valueSet):
									possibleRow.remove(e)
						if possibleRow.__len__() == 2:
							entry = possibleRow[0][0:4] + possibleRow[1][2:4]
							entry.append(list(valueSet))
							squareList.append(entry)
					if possibleCol.__len__() > 2:
						valueSet = []
						for n in possibleCol:
							if n[4].__len__() == 3:
								valueSet = n[4]
								break
						if valueSet.__len__() != 3:
							values = []
							for m in range(possibleCol.__len__()):
								values = values + possibleCol[m][4]
							valueSet = set(values)
							if valueSet.__len__() != 3:
								newSet = []
								for i in valueSet:
									newSet.append(i)
									if values.count(i) < 2:
										newSet.remove(i)
								valueSet = newSet
						if valueSet.__len__() == 3:
							for e in possibleCol:
								if not set(e[4]).issubset(valueSet):
									possibleCol.remove(e)
						if possibleCol.__len__() == 2:
							entry = possibleCol[0][0:4] + possibleCol[1][2:4]
							entry.append(list(valueSet))
							squareList.append(entry)
					if possibleBox.__len__() > 2:
						valueSet = []
						for n in possibleBox:
							if n[4].__len__() == 3:
								valueSet = n[4]
								break
						if valueSet.__len__() != 3:
							values = []
							for m in range(possibleBox.__len__()):
								values = values + possibleBox[m][4]
							valueSet = set(values)
							if valueSet.__len__() != 3:
								newSet = []
								for i in valueSet:
									newSet.append(i)
									if values.count(i) < 2:
										newSet.remove(i)
								valueSet = newSet
						if valueSet.__len__() == 3:
							for e in possibleBox:
								if not set(e[4]).issubset(valueSet):
									possibleBox.remove(e)
						if possibleBox.__len__() == 2:
							entry = possibleBox[0][0:4] + possibleBox[1][2:4]
							entry.append(list(valueSet))
							squareList.append(entry)
		#removes duplicate values from the return list
##		tempList = []
##		for e in squareList:
##			tempList.append(e)
##		for f in tempList:
##			if squareList.count(f) > 1:
##				squareList.remove(f)
##		tempList = []
##		for e in squareList:
##			tempList.append(e)
##		for e in tempList:
##			for f in tempList:
##				if e != f:
##					if e[0] == f[3] and e[1] == f[2] and e[4:] == f[4:]:
##						tempList.remove(e)
##					if e[0] == f[5] and e[1] == f[4] and e[2:4] == f[2:4] and e[6] == f[6]:
##						tempList.remove(e)
##					if e[:2] == f[:2] and e[2] == f[5] and e[3] == f[4] and e[6] == f[6]:
##						tempList.remove(e)
		return squareList

	def findHiddenPair(self):
		"""
			Searches the puzzle for Hidden Pairs
			Only works for repMode 1
			returns list of format:
			[row of square 1, column of square 1, row of square 2, column of square 2, possible values]
		"""
		squareList = []
		if self.repMode == 1:
			self.changeRepMode(2)
			self.findPossiblePuzzle()
			aList = self.findNakedPair()
			self.changeRepMode(3)
			self.findPossiblePuzzle()
			bList = self.findNakedPair()
			self.changeRepMode(1)
			if aList.__len__() > 0:
				for e in aList:
					if e[0] == e[2]:
						temp = []
						for n in e:
							temp.append(n)
						temp[1] = e[4][0] - 1
						temp[4][0] = e[1] + 1
						temp[3] = e[4][1] - 1
						temp[4][1] = e[3] + 1
						squareList.append(temp)
			if bList.__len__() > 0:
				for e in bList:
					if e[1] == e[3]:
						temp = []
						for n in e:
							temp.append(n)
						temp[0] = e[4][0] - 1
						temp[4][0] = e[0] + 1
						temp[2] = e[4][1] - 1
						temp[4][1] = e[2] + 1
						squareList.append(temp)
		return squareList

	def findHiddenTriple(self):
		"""
			Searches the puzzle for hidden triples
			Only works for repMode 1
			returns list of format:
			[row1, column1, row2, column2, row3, column3, pattern values]
		"""
		squareList = []
		if self.repMode == 1:
			self.changeRepMode(2)
			self.findPossiblePuzzle()
			aList = self.findNakedTriple()
			self.changeRepMode(3)
			self.findPossiblePuzzle()
			bList = self.findNakedTriple()
			self.changeRepMode(1)
			if aList.__len__() > 0:
				for e in aList:
					if e[0] == e[2] and e[0] == e[4]:
						temp = []
						for n in e:
							temp.append(n)
						temp[0] = e[6][0] - 1
						temp[6][0] = e[0] + 1
						temp[2] = e[6][1] - 1
						temp[6][1] = e[2] + 1
						temp[4] = e[6][2] - 1
						temp[6][2] = e[4] + 1
						squareList.append(temp)
			if bList.__len__() > 0:
				for e in bList:
					if e[1] == e[3] and e[1] == e[5]:
						temp = []
						for n in e:
							temp.append(n)
						temp[1] = e[6][0] - 1
						temp[6][0] = e[1] + 1
						temp[3] = e[6][1] - 1
						temp[6][1] = e[3] + 1
						temp[5] = e[6][2] - 1
						temp[6][2] = e[5] + 1
						squareList.append(temp)
		return squareList

	def findXWing(self):    #needs refinement
		"""
			Searches the puzzle for X-Wing patterns
			Only works for repMode1
			known bug: It can return patterns which aren't quite x-wings but are very close
			returns list of squares with format:
			[row1, column1, row2 column2, row3, column3, row4, column4, pattern value]
		"""
		squareList = []
		if self.repMode == 1:
			self.changeRepMode(2)
			self.findPossiblePuzzle()
			aList = self.findNakedPair()
##			self.changeRepMode(3)
##			self.findPossiblePuzzle()
##			bList = self.findNakedPair()
			self.changeRepMode(1)
			if aList.__len__() > 0:
				for e in aList:
					if e[1] == e[3]:
						temp = []
						for n in e:
							temp.append(n)
						temp[1] = e[4][0] - 1
						temp[3] = e[4][1] - 1
						temp[4] = e[0]
						temp.append(e[4][1] - 1)
						temp.append(e[2])
						temp.append(e[4][0] - 1)
						temp.append(e[1] + 1)
						squareList.append(temp)
##			if bList.__len__() > 0:
##				for e in bList:
##					if e[0] == e[2]:
##						temp = []
##						for n in e:
##							temp.append(n)
##						temp[0] = e[4][0] - 1
##						temp[2] = e[4][1] - 1
##						temp[4] = e[1]
##						temp.append(e[4][1] - 1)
##						temp.append(e[3])
##						temp.append(e[4][0] - 1)
##						temp.append(e[0] + 1)
##						squareList.append(temp)
		return squareList

	def findSwordfish(self):
		"""
			Searches the puzzle for swordfish patterns
			Only works for repMode 1
			returns a list of squares with format:
			[row1, column1, row2, column2, row3, column3, row4, column4, row5, column5, row6, column6, pattern value]
		"""
		squareList = []
		tempList = []
		if self.repMode == 1:
			self.changeRepMode(2)
			self.findPossiblePuzzle()
			aList = self.findNakedTriple()
##			self.changeRepMode(3)
##			self.findPossiblePuzzle()
##			bList = self.findNakedTriple()
			self.changeRepMode(1)
			if aList.__len__() > 0:
				for e in aList:
					if e[1] == e[3] and e[1] == e[5]:
						temp = []
						for n in e:
							temp.append(n)
						temp[1] = e[6][0] - 1
						temp[3] = e[6][1] - 1
						temp[5] = e[6][2] - 1
						temp[6] = e[0]
						temp.append(e[6][1] - 1)
						temp.append(e[2])
						temp.append(e[6][2] - 1)
						temp.append(e[4])
						temp.append(e[6][0] - 1)
						temp.append(e[0])
						temp.append(e[6][2] - 1)
						temp.append(e[2])
						temp.append(e[6][0] - 1)
						temp.append(e[4])
						temp.append(e[6][1] - 1)
						temp.append(e[1] + 1)
						squareList.append(temp)
##			if bList.__len__() > 0:
##				for e in bList:
##					if e[0] == e[2] and e[0] == e[4]:
##						temp = []
##						for n in e:
##							temp.append(n)
##						temp[0] = e[6][0] - 1
##						temp[6][0] = e[0] + 1
##						temp[2] = e[6][1] - 1
##						temp[6][1] = e[2] + 1
##						temp[4] = e[6][2] - 1
##						temp[6][2] = e[4] + 1
##						squareList.append(temp)
##			self.findPossiblePuzzle()
##			for e in tempList:
##				temp = []
##				popList = []
##				for n in e:
##					temp.append(n)
##				for n in range(0, e.__len__(), 2):
##					if self.possible[e[n]][e[n+1]].count(e[e.__len__()-1]) != 1:
##						popList.append(n)
##						popList.append(n+1)
##				popList.reverse()
##				for i in popList:
##					temp.pop(i)
##				squareList.append(temp)
		return squareList
			
#was originally going to search for multiple lines pattern but ran out of time to implement it - further work

##	def findMultipleLines(self):	
##		"""
##			Finds the multiple lines paattern
##		"""
##		squareList = []
##		x = 0
##		while (x < (self.SIZE - 1)):		#Checks the different boxes of Sudoku
##			y = 0
##			while (y < (self.SIZE - 1)):	#while x and y are less than index of last position of list
##				for value in range(self.SIZE + 1):	#checks each possible value
##					possibleRow = []
##					possibleCol = []
##					for n in range(x, x + self.BOX_WIDTH):
##						for i in range(y, y + self.BOX_WIDTH):
##							if self.possible[n][i].count(value) > 0:
##								possibleRow.append(n)
##							if self.possible[i][n].count(value) > 0:
##								possibleCol.append(n)
##					if possibleRow.__len__() > 0:
##						if possibleRow.count(possibleRow[0]) == possibleRow.__len__():
##							squareList.append([1, x, y, possibleRow[0], value])
##					if possibleCol.__len__() > 0:
##						if possibleCol.count(possibleCol[0]) == possibleCol.__len__():
##							squareList.append([2, x, y, possibleCol[0], value])
##				y = y + (self.BOX_WIDTH)
##			x = x + (self.BOX_WIDTH)
##		return squareList
##				

	def updatePuzzle(self):
		"""
			Updates each of the different representations of the Sudoku based on the workable puzzle
		"""
		try:	
			if self.repMode == 1:	#puzzle is RowColNum
				self.puzzleRowColNum = self.puzzle
				self.generateNumColRow()
				self.generateRowNumCol()
			elif self.repMode == 2:	#puzzle is RowNumCol
				self.puzzleRowNumCol = self.puzzle
				for row in range(self.SIZE):
					for value in range(self.SIZE):
						col = self.puzzle[row][value]
						if col > 0:
							self.puzzleRowColNum[row][col - 1] = value + 1
				self.generateNumColRow()
			elif self.repMode == 3:	#puzzle is NumColRow
				self.puzzleNumColRow = self.puzzle
				for value in range(self.SIZE):
					for col in range(self.SIZE):
						row = self.puzzle[value][col]
						if row > 0:
							self.puzzleRowColNum[row - 1][col] = value + 1
				self.generateRowNumCol()
			else:		#puzzle is invalid
				raise Exception("Error")
		except Exception as e:
			print e

	def changeRepMode(self, mode):
		"""
			Changes the representation mode of the puzzle
			mode 1 is traditional RowColNum representation
			mode 2 is Row on the y axis, Number on the x axis and Columns in the field - RowNumCol
			mode 3 is Number on the y axis, Column on the x axis and Rows in the field - NumColRow
		"""
		if mode > 0 and mode <= 3:
			self.updatePuzzle()
			self.repMode = mode	#display mode
			if mode == 1:				
				self.puzzle = self.puzzleRowColNum
			elif mode == 2:
				self.puzzle = self.puzzleRowNumCol
			else:
				self.puzzle = self.puzzleNumColRow
		else:
			raise Exception("Invalid display Mode")
